use util:serialize;
use core:io;
use parser;

/**
 * Misc. data types.
 */


/**
 * Representation of a single problem (no source code, intended for lists of problems).
 */
class ProblemInfo : serializable {
	Int id;

	Str title;
	Str author;

	// Is the next step to modify the implementation?
	Bool hasError;

	// Does the current user have any solutions to this problem?
	Bool attempted;

	init(Int id, Str title, Str author, Bool hasError, Bool attempted) {
		init {
			id = id;
			title = title;
			author = author;
			hasError = hasError;
			attempted = attempted;
		}
	}

	// For sorting.
	Bool <(ProblemInfo o) {
		id < o.id;
	}

	// To string.
	void toS(StrBuf to) : override {
		to << "{id: " << id << ", title: " << title << ", author: " << author << ", hasError: " << hasError << ", attempted: " << attempted << "}";
	}
}


/**
 * Representation of something that we can attempt to solve. Includes the code necessary for the
 * client to solve the problem. Also contains information about the origins of the source code so
 * that a solution can be submitted.
 */
class Problem : serializable {
	// ID of the problem we are solving.
	Int id;

	// Title of the problem.
	Str title;

	// If positive: ID of the implementation we are using.
	Int implId;

	// Version of the implementation.
	Int implVersion;

	// If positive: ID of the test we are using.
	Int testId;

	// Version of the test.
	Int testVersion;

	// Does the implementation + test expose an error?
	Error error;

	// Implementation.
	Code impl;

	// Test program.
	Code test;

	// Reference implementation.
	Code ref;

	// Create.
	init(Int id, Str title, Int implId, Int implVer, Int testId, Int testVer, Error error, Code impl, Code test, Code ref) {
		init {
			id = id;
			title = title;
			implId = implId;
			implVersion = implVer;
			testId = testId;
			testVersion = testVer;
			error = error;
			impl = impl;
			test = test;
			ref = ref;
		}
	}
}


/**
 * A piece of code, indicating a language as well.
 */
class Code : serializable {
	// ID in the database.
	Int id;

	// Source code.
	Str src;

	// Language (= file extension).
	Str language;

	// Create.
	init(Str src, Str language) {
		init {
			id = -1;
			src = src;
			language = language;
		}
	}

	// Create.
	init(Int id, Str src, Str language) {
		init {
			id = id;
			src = src;
			language = language;
		}
	}

	// Add to a MemoryProtocol.
	Url put(Str base, MemoryProtocol to) {
		MemOStream out;
		Utf8Output text(out);
		text.write(src);
		text.flush();
		to.put(base + "." + language, out.buffer);
	}

	// Produce a "signature" for this code.
	Str signature() {
		codeSignature(src);
	}
}

/**
 * Type of improvement: test or implementation.
 */
enum ImprovementType {
	test,
	implementation,
}


/**
 * Improvement to a problem. Either a test or an implementation, depending on context.
 */
class Improvement : serializable {
	// ID of this improvement.
	Int id;

	// Version of the improvement.
	Int version;

	// Author.
	Str author;

	// Error?
	Error error;

	// Create.
	init(Int id, Int version, Str author, Error error) {
		init {
			id = id;
			version = version;
			author = author;
			error = error;
		}
	}
}

/**
 * Error status of a problem.
 * Represents what type of error a problem has, and if the status is unknown.
 */
value Error : serializable {
	// Error message, if any. Empty string means that all is well. Null means unknown.
	private Str? data;

	// Internal helper.
	private init(Str? data) {
		init { data = data; }
	}

	Error error(Str type) : static { Error(type); }
	Error success() : static { Error(""); }
	Error unknown() : static { Error(null); }

	// Is it a success?
	Bool success() {
		if (data)
			return data.empty;
		false;
	}

	// Is it an error?
	Str? error() {
		if (data)
			if (data.any)
				return data;
		return null;
	}

	// Is it unknown?
	Bool unknown() {
		data.empty;
	}

	// Do we have an error?
	Bool any() {
		error.any;
	}

	Str toS() {
		if (data) {
			if (data.empty) {
				return "No error";
			} else {
				return "Error: " + data;
			}
		} else {
			return "Unknown";
		}
	}
}

/**
 * A single high-score entry.
 */
class Score : serializable {
	Str name;
	Int points;
	Int place;

	init(Str name, Int points, Int place) {
		init {
			name = name;
			points = points;
			place = place;
		}
	}
}

// Create a "signature" that allows comparing code for equality.
Str codeSignature(Str code) {
	sigParser(code).value.toS;
}

// Remove whitespace from a string to create a "hash" of a program. Note: This is not 100%
// accurate, but good enough to detect "significant" changes (for example "myclass a" and "my
// classa" will hash to the same thing.
sigParser : parser(recursive descent) {
	start = Start;

	StrBuf Start();
	Start => StrBuf() : "[ \n\r\t]*" - ("[^ \n\r\t]+" -> add - "[ \n\r\t]*")*;
}
