/*
 * Copyright (c) 2023, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */


import java.io.BufferedReader;
import java.io.IOException;
import java.nio.file.Path;
import java.util.List;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import java.util.logging.StreamHandler;
import java.util.stream.Stream;


import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.junit.jupiter.params.ParameterizedTest;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.fail;

/*
 * @test
 * @summary verify logging of call to System.exit or Runtime.exit.
 * @run junit/othervm RuntimeExitLogTest
 */

public class RuntimeExitLogTest {

    private static final String TEST_JDK = System.getProperty("test.jdk");
    private static final String TEST_SRC = System.getProperty("test.src");

    private static Object HOLD_LOGGER;

    /**
     * Call System.exit() with the parameter (or zero if not supplied).
     * @param args zero or 1 argument, an exit status
     */
    public static void main(String[] args) throws InterruptedException {
        int status = args.length > 0 ? Integer.parseInt(args[0]) : 0;
        if (System.getProperty("ThrowingHandler") != null) {
            HOLD_LOGGER = ThrowingHandler.installHandler();
        }
        System.exit(status);
    }

    /**
     * Test various log level settings, and none.
     * @return a stream of arguments for parameterized test
     */
    private static Stream<Arguments> logParamProvider() {
        return Stream.of(
                // Logging enabled with level DEBUG
                Arguments.of(List.of("-Djava.util.logging.config.file=" +
                        Path.of(TEST_SRC, "ExitLogging-FINE.properties").toString()), 1,
                        "Runtime.exit() called with status: 1"),
                // Logging disabled due to level
                Arguments.of(List.of("-Djava.util.logging.config.file=" +
                        Path.of(TEST_SRC, "ExitLogging-INFO.properties").toString()), 2,
                        ""),
                // Console logger
                Arguments.of(List.of("--limit-modules", "java.base",
                        "-Djdk.system.logger.level=DEBUG"), 3,
                        "Runtime.exit() called with status: 3"),
                // Console logger
                Arguments.of(List.of(), 4, ""),
                // Throwing Handler
                Arguments.of(List.of("-DThrowingHandler",
                        "-Djava.util.logging.config.file=" +
                        Path.of(TEST_SRC, "ExitLogging-FINE.properties").toString()), 5,
                        "Runtime.exit(5) logging failed: Exception in publish")
                );
    }

    /**
     * Check that the logger output of a launched process contains the expected message.
     * @param logProps The name of the log.properties file to set on the command line
     * @param status the expected exit status of the process
     * @param expectMessage log should contain the message
     */
    @ParameterizedTest
    @MethodSource("logParamProvider")
    public void checkLogger(List<String> logProps, int status, String expectMessage) {
        ProcessBuilder pb = new ProcessBuilder();
        pb.redirectErrorStream(true);

        List<String> cmd = pb.command();
        cmd.add(Path.of(TEST_JDK,"bin", "java").toString());
        cmd.addAll(logProps);
        cmd.add(this.getClass().getName());
        cmd.add(Integer.toString(status));

        try {
            Process process = pb.start();
            try (BufferedReader reader = process.inputReader()) {
                List<String> lines = reader.lines().toList();
                boolean match = (expectMessage.isEmpty())
                        ? lines.size() == 0
                        : lines.stream().filter(s -> s.contains(expectMessage)).findFirst().isPresent();
                if (!match) {
                    // Output lines for debug
                    System.err.println("Expected: \"" + expectMessage + "\"");
                    System.err.println("---- Actual output begin");
                    lines.forEach(l -> System.err.println(l));
                    System.err.println("---- Actual output end");
                    fail("Unexpected log contents");
                }
            }
            int result = process.waitFor();
            assertEquals(status, result, "Exit status");
        } catch (IOException | InterruptedException ex) {
            fail(ex);
        }
    }

    /**
     * A LoggingHandler that throws an Exception.
     */
    public static class ThrowingHandler extends StreamHandler {

        // Install this handler for java.lang.Runtime
        public static Logger installHandler() {
            Logger logger = Logger.getLogger("java.lang.Runtime");
            logger.addHandler(new ThrowingHandler());
            return logger;
        }

        @Override
        public synchronized void publish(LogRecord record) {
            super.publish(record);
            throw new RuntimeException("Exception in publish");
        }
    }
}
