# Copyright © 2008 Ian Jackson <ijackson@chiark.greenend.org.uk>
# Copyright © 2008 Canonical, Ltd.
#   written by Colin Watson <cjwatson@ubuntu.com>
# Copyright © 2008 James Westby <jw+debian@jameswestby.net>
# Copyright © 2009 Raphaël Hertzog <hertzog@debian.org>
# Copyright © 2022 Ruben Rodriguez <ruben@trisquel.org>
# Copyright © 2024 Luis Guzman <ark@switnet.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.

=encoding utf8

=head1 NAME

Dpkg::Vendor::Trisquel - Trisquel vendor class

=head1 DESCRIPTION

This vendor class customizes the behavior of dpkg scripts for Trisquel
specific behavior and policies.

B<Note>: This is a private module, its API can change at any time.

=cut

package Dpkg::Vendor::Trisquel 0.01;

use strict;
use warnings;

use List::Util qw(any);

use Dpkg::ErrorHandling;
use Dpkg::Gettext;
use Dpkg::Control::Types;

use parent qw(Dpkg::Vendor::Debian);

sub run_hook {
    my ($self, $hook, @params) = @_;

    if ($hook eq 'package-keyrings') {
        return ($self->SUPER::run_hook($hook),
                '/usr/share/keyrings/trisquel-archive-keyring.gpg');
    } elsif ($hook eq 'archive-keyrings') {
        return ($self->SUPER::run_hook($hook),
                '/usr/share/keyrings/trisquel-archive-keyring.gpg');
    } elsif ($hook eq 'archive-keyrings-historic') {
        return ($self->SUPER::run_hook($hook),
                '/usr/share/keyrings/trisquel-archive-removed-keys.gpg');
    } else {
        return $self->SUPER::run_hook($hook, @params);
    }
    # Default return value for unknown/unimplemented hooks
    return;
}

sub _lto_disabled {
    my $fn = "/usr/share/lto-disabled-list/lto-disabled-list";
    open(LIST, "<", $fn) or return;

    # get source name
    -r "debian/control" or return;
    require Dpkg::Control::Info;
    my $ctrl = Dpkg::Control::Info->new();
    my $src_fields = $ctrl->get_source();
    return unless defined $src_fields;

    my $src = "";
    foreach (keys %{$src_fields}) {
        my $v = $src_fields->{$_};
        if (m/^Source$/i) {
            $src = $v;
            last;
        }
    }
    return unless $src ne "";

    my $arch = Dpkg::Arch::get_host_arch();

    # read disabled-list
    while (<LIST>) {
        if (m/^$src\s/) {
            if (m/^$src\s.*(any|$arch)\s/) {
                close(LIST);
                return 1;
            } else {
                close(LIST);
                return;
            }
        }
    }
    close(LIST);
    return;
}

# Override Debian default features.
sub init_build_features {
    my ($self, $use_feature, $builtin_feature) = @_;

    $self->SUPER::init_build_features($use_feature, $builtin_feature);

    require Dpkg::Arch;
    my $arch = Dpkg::Arch::get_host_arch();

    if (any { $_ eq $arch } qw(amd64 arm64 ppc64el s390x)) {
        $use_feature->{optimize}{lto} = 1;
	if (_lto_disabled()) {
        	$use_feature->{optimize}{lto} = 0;
	}
    }

    if (any { $_ eq $arch } qw(amd64 arm64 riscv64 s390x)) {
        $use_feature->{qa}{framepointer} = 1;
    }
}

sub set_build_features {
    my ($self, $flags) = @_;

    $self->SUPER::set_build_features($flags);

    require Dpkg::Arch;
    my $arch = Dpkg::Arch::get_host_arch();

    if ($arch eq 'ppc64el' && $flags->get_option_value('optimize-level') != 0) {
        $flags->set_option_value('optimize-level', 3);
    }

    $flags->set_option_value('fortify-level', 3);

    # Debian enables -fstack-clash-protection but it causes troubles on armhf
    # (which are not fully analyzed as of writing this); disable it there in
    # order to have time to investigate
    if ($arch eq 'armhf') {
        # Stack clash protector only available on amd64 and arm.
        $flags->set_feature('hardening', 'stackclash', 0);
    }
}

sub add_build_flags {
    my ($self, $flags) = @_;

    my @compile_flags = qw(
        CFLAGS
        CXXFLAGS
        OBJCFLAGS
        OBJCXXFLAGS
        FFLAGS
        FCFLAGS
    );

    $self->SUPER::add_build_flags($flags);

    # Per https://wiki.ubuntu.com/DistCompilerFlags
    $flags->prepend('LDFLAGS', '-Wl,-Bsymbolic-functions');

    # Add Rust flags
    $flags->append('RUSTFLAGS', '');
    $flags->append('RUSTFLAGS_FOR_BUILD', '');
    # Rust frame pointer switch
    if ($flags->use_feature('qa', 'framepointer')) {
        my $arch = Dpkg::Arch::get_host_arch();
        if (Dpkg::Arch::debarch_eq($arch, 's390x')) {
            $flags->append('RUSTFLAGS', '-Ctarget-feature=+backchain');
        } else {
            $flags->append('RUSTFLAGS', '-Cforce-frame-pointers=yes');
        }
    }

    # In Ubuntu these flags are set by the compiler, so when disabling the
    # features we need to pass appropriate flags to disable them.
    if (!$flags->use_feature('hardening', 'stackprotectorstrong') &&
        !$flags->use_feature('hardening', 'stackprotector')) {
        my $flag = '-fno-stack-protector';
        $flags->append($_, $flag) foreach @compile_flags;
    }

    if (!$flags->use_feature('hardening', 'stackclash')) {
        my $flag = '-fno-stack-clash-protection';
        $flags->append($_, $flag) foreach @compile_flags;
    }

    if (!$flags->use_feature('hardening', 'fortify')) {
        $flags->append('CPPFLAGS', '-D_FORTIFY_SOURCE=0');
    }

    if (!$flags->use_feature('hardening', 'format')) {
        my $flag = '-Wno-format -Wno-error=format-security';
        $flags->append('CFLAGS', $flag);
        $flags->append('CXXFLAGS', $flag);
        $flags->append('OBJCFLAGS', $flag);
        $flags->append('OBJCXXFLAGS', $flag);
     }

     if (!$flags->use_feature('hardening', 'branch')) {
        my $cpu = $flags->get_option_value('hardening-branch-cpu');
        my $flag;
        if ($cpu eq 'arm64') {
            $flag = '-mbranch-protection=none';
        } elsif ($cpu eq 'amd64') {
            $flag = '-fcf-protection=none';
        }
        if (defined $flag) {
            $flags->append($_, $flag) foreach @compile_flags;
        }
    }

    # We always enable fdebug-prefix-map and (ideally) set the new
    # path as full (unless ${DEB_BUILD_DEBUGPATH} points to a relative
    # path), because (a) the DWARF spec forbids relative paths (which
    # gets mapped to DW_AT_comp_dir), and (b) it makes our debuginfod
    # service happy when indexing source code.
    if ($flags->use_feature('reproducible', 'fixfilepath')) {
        my $build_path = $flags->get_option_value('build-path');
        my $stripflag = '-fdebug-prefix-map=' . $build_path . '=.';

        if (defined $ENV{DEB_BUILD_DEBUGPATH}) {
            my $debugprefixmap = '-fdebug-prefix-map=' . $build_path . '=' . $ENV{DEB_BUILD_DEBUGPATH};

            # Strip any existing -fdebug-prefix-map flag.
            $flags->strip($_, $stripflag) foreach @compile_flags;
            $flags->append($_, $debugprefixmap) foreach @compile_flags;
        } elsif (-r 'debian/changelog') {
            require Dpkg::Changelog::Debian;
            my $pkgchangelog = Dpkg::Changelog::Debian->new(range => { "count" => 1 });
            $pkgchangelog->load('debian/changelog');
            my $chgentry = @{$pkgchangelog}[0];
            my $pkgver = $chgentry->get_version();
            my $pkgsrc = $chgentry->get_source();

            if ($pkgver ne "" && $pkgsrc ne "") {
                my $debugprefixmap = '-fdebug-prefix-map=' . $build_path . '=/usr/src/' . $pkgsrc . '-' . $pkgver;

                # Strip any existing -fdebug-prefix-map flag.
                $flags->strip($_, $stripflag) foreach @compile_flags;
                $flags->append($_, $debugprefixmap) foreach @compile_flags;
            }
        }
    }

    return;
}

=head1 CHANGES

=head2 Version 0.xx

This is a private module.

=cut

1;
